// ---------- REQUIRED LIBRARIES ----------
#include <Wire.h>                  // Enables I2C communication (for MPU6050 and OLED)
#include <Adafruit_SSD1306.h>      // Controls SSD1306 OLED screen via I2C
#include <Adafruit_GFX.h>          // Base graphics library used by SSD1306
#include <MPU6050.h>               // Library for MPU6050 sensor (accelerometer + gyroscope)

// ---------- CUSTOM LOGO (optional) ----------
extern const unsigned char myLogo[];  // 128x64 pixel logo to display on startup

// ---------- OLED DISPLAY PARAMETERS ----------
#define SCREEN_WIDTH 128           // Screen width in pixels
#define SCREEN_HEIGHT 64           // Screen height in pixels
#define SCREEN_ADDRESS 0x3C        // Standard I2C address for SSD1306
#define OLED_RESET -1              // No reset pin used (not needed for I2C)

// ---------- OLED DISPLAY OBJECT ----------
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---------- MPU6050 SENSOR OBJECT ----------
MPU6050 mpu;  // 6-axis inertial sensor: 3 accelerometers + 3 gyroscopes

// ---------- INITIALIZATION FUNCTION ----------
void setup()
{
  Serial.begin(115200);      // Start serial monitor at 115200 baud
  Wire.begin();              // Start I2C bus
  mpu.initialize();          // Initialize MPU6050 sensor

  // Initialize OLED screen
  if (!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
    Serial.println("OLED initialization failed!"); // Error message if screen not detected
    while (true); // Infinite loop to block the program
  }

  // Show logo at startup for 1 second
  display.clearDisplay();                         // Clear the screen
  display.drawBitmap(0, 0, myLogo, 128, 64, WHITE); // Display logo
  display.setTextSize(1);                         // Standard text size
  display.setTextColor(SSD1306_WHITE);            // White text
  display.display();                              // Refresh screen
  delay(1000);                                     // Pause for 1 second
}

// ---------- MAIN LOOP ----------
void loop()
{
  // Variables for raw sensor data (acceleration on X, Y, Z)
  int16_t ax, ay, az;

  // Variables for normalized values (in g) and converted (in m/s²)
  float axf_g, ayf_g, azf_g;         // Normalized accelerations (g)
  float axf_ms2, ayf_ms2, azf_ms2;   // Converted accelerations (m/s²)

  // Orientation variables (in degrees)
  float pitch, roll;

  // Read raw acceleration data from sensor
  mpu.getAcceleration(&ax, &ay, &az);

  // Convert raw values to "g" (1 g ≈ 16384 LSB for ±2g range)
  axf_g = ax / 16384.0;
  ayf_g = ay / 16384.0;
  azf_g = az / 16384.0;

  // Convert values to m/s² (1 g = 9.81 m/s²)
  axf_ms2 = axf_g * 9.81;
  ayf_ms2 = ayf_g * 9.81;
  azf_ms2 = azf_g * 9.81;

  // Calculate tilt angles in degrees (trigonometric formulas)
  pitch = atan2(axf_g, sqrt(ayf_g * ayf_g + azf_g * azf_g)) * 180.0 / PI; // Forward/backward tilt
  roll  = atan2(ayf_g, sqrt(axf_g * axf_g + azf_g * azf_g)) * 180.0 / PI; // Left/right tilt

  // Calculate total acceleration (magnitude of vector a) in m/s²
  float a_total = sqrt(axf_ms2 * axf_ms2 + ayf_ms2 * ayf_ms2 + azf_ms2 * azf_ms2);

  // Detect hand position based on pitch/roll/az thresholds
  String orientation = [&]() {
    if (pitch > 30)
    {
      return "Front";     // Tilted forward
    }
    if (pitch < -30)
    {
      return "Back";     // Tilted backward
    }
    if (roll > 30)
    {
      return "Right";      // Tilted to the right
    }
    if (roll < -30)
    {
      return "Left";      // Tilted to the left
    }
    if (azf_g > 0.85)
    {
      return "Up";      // Palm facing up
    }
    if (azf_g < -0.85)
    {
      return "Down";   // Palm facing down
    }
    return "Stable";   // No significant movement
  }();

  // ---------- OLED DISPLAY ----------
  display.clearDisplay();               // Clear the screen
  display.setCursor(0, 0);              // Cursor position
  display.println("MPU6050 Orientation"); // Title

  // Display detected orientation
  display.setCursor(0, 20);
  display.print("Orientation: ");
  display.println(orientation);

  // Display converted accelerations (m/s²)
  display.print("ax: ");
  display.print(axf_ms2, 2); display.println(" m/s2");

  display.print("ay: ");
  display.print(ayf_ms2, 2); display.println(" m/s2");

  display.print("az: ");
  display.print(azf_ms2, 2); display.println(" m/s2");

  // Display total motion intensity
  display.print("a: ");
  display.print(a_total, 2); display.println(" m/s2");

  display.display(); // Refresh the display

  // ---------- SERIAL MONITOR OUTPUT ----------
  Serial.println("=========={EPITECH.}===========");
  Serial.print("Orientation: "); Serial.println(orientation);
  Serial.print("ax: "); Serial.print(axf_ms2); Serial.println(" m/s2");
  Serial.print("ay: "); Serial.print(ayf_ms2); Serial.println(" m/s2");
  Serial.print("az: "); Serial.print(azf_ms2); Serial.println(" m/s2");
  Serial.print("a : "); Serial.print(a_total); Serial.println(" m/s2");

  delay(200); // Wait 200 ms (5 measurements per second)
}
